<?php
// Database connection
$host = 'localhost:3306';
$dbname = 'tawasoll_book';
$username = 'root'; // Change to your username
$password = ''; // Change to your password

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    die("Connection failed: " . $e->getMessage());
}

// Get all customers for dropdown
$customers_query = "SELECT customer_phone, name, business_type, current_balance FROM customer_context ORDER BY name";
$customers = $pdo->query($customers_query)->fetchAll(PDO::FETCH_ASSOC);

// Filter parameters
$selected_customer = isset($_GET['customer']) ? $_GET['customer'] : '';
$date_from = isset($_GET['date_from']) ? $_GET['date_from'] : '';
$date_to = isset($_GET['date_to']) ? $_GET['date_to'] : '';
$search = isset($_GET['search']) ? $_GET['search'] : '';
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$per_page = 20;
$offset = ($page - 1) * $per_page;

// Build query
$where_conditions = [];
$params = [];

if ($selected_customer) {
    $where_conditions[] = "t.customer_phone = :customer_phone";
    $params[':customer_phone'] = $selected_customer;
}

if ($date_from) {
    $where_conditions[] = "DATE(t.transaction_date) >= :date_from";
    $params[':date_from'] = $date_from;
}

if ($date_to) {
    $where_conditions[] = "DATE(t.transaction_date) <= :date_to";
    $params[':date_to'] = $date_to;
}

if ($search) {
    $where_conditions[] = "(t.description LIKE :search OR t.notes LIKE :search OR t.foreign_currency LIKE :search)";
    $params[':search'] = "%$search%";
}

$where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

// Get transactions
$query = "SELECT t.*, c.name as customer_name, c.business_type, c.current_balance as customer_balance
          FROM transactions t
          LEFT JOIN customer_context c ON t.customer_phone = c.customer_phone
          $where_clause
          ORDER BY t.transaction_date DESC, t.id DESC
          LIMIT :limit OFFSET :offset";

$stmt = $pdo->prepare($query);
foreach ($params as $key => $value) {
    $stmt->bindValue($key, $value);
}
$stmt->bindValue(':limit', $per_page, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();
$transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get total count
$count_query = "SELECT COUNT(*) as total FROM transactions t $where_clause";
$count_stmt = $pdo->prepare($count_query);
foreach ($params as $key => $value) {
    $count_stmt->bindValue($key, $value);
}
$count_stmt->execute();
$total_records = $count_stmt->fetch(PDO::FETCH_ASSOC)['total'];
$total_pages = ceil($total_records / $per_page);

// Calculate summary
$summary_query = "SELECT 
                    SUM(CASE WHEN transaction_type = 'CREDIT' THEN aed_amount ELSE 0 END) as total_credits,
                    SUM(CASE WHEN transaction_type = 'DEBIT' THEN aed_amount ELSE 0 END) as total_debits,
                    COUNT(*) as transaction_count
                  FROM transactions t
                  $where_clause";
$summary_stmt = $pdo->prepare($summary_query);
foreach ($params as $key => $value) {
    $summary_stmt->bindValue($key, $value);
}
$summary_stmt->execute();
$summary = $summary_stmt->fetch(PDO::FETCH_ASSOC);

// Get customer info if selected
$customer_info = null;
if ($selected_customer) {
    $customer_stmt = $pdo->prepare("SELECT * FROM customer_context WHERE customer_phone = :phone");
    $customer_stmt->execute([':phone' => $selected_customer]);
    $customer_info = $customer_stmt->fetch(PDO::FETCH_ASSOC);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Customer Bookkeeping - Tawasoll</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        body {
            background-color: #f8f9fa;
        }
        .header-section {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 2rem 0;
            margin-bottom: 2rem;
        }
        .summary-card {
            border-left: 4px solid #667eea;
            transition: transform 0.2s;
        }
        .summary-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }
        .credit-row {
            background-color: #d4edda;
        }
        .debit-row {
            background-color: #f8d7da;
        }
        .customer-info-card {
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            border-radius: 10px;
            padding: 1.5rem;
            margin-bottom: 1.5rem;
        }
        .action-btn {
            padding: 0.25rem 0.5rem;
            font-size: 0.875rem;
            margin: 0 2px;
        }
        .table-container {
            background: white;
            border-radius: 10px;
            padding: 1.5rem;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }
        .filter-section {
            background: white;
            border-radius: 10px;
            padding: 1.5rem;
            margin-bottom: 1.5rem;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }
    </style>
</head>
<body>
    <div class="header-section">
        <div class="container">
            <h1 class="mb-0"><i class="bi bi-book"></i> Customer Bookkeeping System</h1>
            <p class="mb-0">Tawasoll Book - Transaction Management</p>
        </div>
    </div>

    <div class="container">
        <!-- Filter Section -->
        <div class="filter-section">
            <form method="GET" action="" id="filterForm">
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label"><i class="bi bi-person"></i> Customer</label>
                        <select name="customer" class="form-select" id="customerSelect">
                            <option value="">All Customers</option>
                            <?php foreach ($customers as $customer): ?>
                                <option value="<?= htmlspecialchars($customer['customer_phone']) ?>" 
                                        <?= $selected_customer == $customer['customer_phone'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($customer['name'] ?: $customer['customer_phone']) ?> 
                                    (<?= htmlspecialchars($customer['business_type'] ?: 'N/A') ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label"><i class="bi bi-calendar"></i> From Date</label>
                        <input type="date" name="date_from" class="form-control" value="<?= htmlspecialchars($date_from) ?>">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label"><i class="bi bi-calendar"></i> To Date</label>
                        <input type="date" name="date_to" class="form-control" value="<?= htmlspecialchars($date_to) ?>">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label"><i class="bi bi-search"></i> Search</label>
                        <input type="text" name="search" class="form-control" placeholder="Description, notes..." 
                               value="<?= htmlspecialchars($search) ?>">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">&nbsp;</label>
                        <div class="d-grid gap-2">
                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-filter"></i> Filter
                            </button>
                            <a href="customer_bookkeeping.php" class="btn btn-secondary">
                                <i class="bi bi-arrow-clockwise"></i> Reset
                            </a>
                        </div>
                    </div>
                </div>
            </form>
        </div>

        <!-- Customer Info Card -->
        <?php if ($customer_info): ?>
        <div class="customer-info-card">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <h4 class="mb-2"><i class="bi bi-person-circle"></i> <?= htmlspecialchars($customer_info['name'] ?: 'N/A') ?></h4>
                    <div class="row">
                        <div class="col-md-4">
                            <strong>Phone:</strong> <?= htmlspecialchars($customer_info['customer_phone']) ?>
                        </div>
                        <div class="col-md-4">
                            <strong>Business:</strong> <?= htmlspecialchars($customer_info['business_type'] ?: 'N/A') ?>
                        </div>
                        <div class="col-md-4">
                            <strong>Preferred Currency:</strong> <?= htmlspecialchars($customer_info['preferred_currency'] ?: 'N/A') ?>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 text-end">
                    <h5>Current Balance</h5>
                    <h2 class="mb-0 <?= $customer_info['current_balance'] >= 0 ? 'text-success' : 'text-danger' ?>">
                        <?= number_format($customer_info['current_balance'], 2) ?> AED
                    </h2>
                    <small>Transactions: <?= number_format($customer_info['transaction_count']) ?></small>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Summary Cards -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card summary-card">
                    <div class="card-body">
                        <h6 class="card-subtitle mb-2 text-muted"><i class="bi bi-arrow-down-circle"></i> Total Credits</h6>
                        <h3 class="card-title text-success mb-0"><?= number_format($summary['total_credits'] ?? 0, 2) ?> AED</h3>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card summary-card">
                    <div class="card-body">
                        <h6 class="card-subtitle mb-2 text-muted"><i class="bi bi-arrow-up-circle"></i> Total Debits</h6>
                        <h3 class="card-title text-danger mb-0"><?= number_format($summary['total_debits'] ?? 0, 2) ?> AED</h3>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card summary-card">
                    <div class="card-body">
                        <h6 class="card-subtitle mb-2 text-muted"><i class="bi bi-calculator"></i> Net Balance</h6>
                        <h3 class="card-title <?= ($summary['total_credits'] - $summary['total_debits']) >= 0 ? 'text-success' : 'text-danger' ?> mb-0">
                            <?= number_format(($summary['total_credits'] ?? 0) - ($summary['total_debits'] ?? 0), 2) ?> AED
                        </h3>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card summary-card">
                    <div class="card-body">
                        <h6 class="card-subtitle mb-2 text-muted"><i class="bi bi-list-ol"></i> Transactions</h6>
                        <h3 class="card-title text-primary mb-0"><?= number_format($summary['transaction_count'] ?? 0) ?></h3>
                    </div>
                </div>
            </div>
        </div>

        <!-- Export Buttons -->
        <div class="d-flex justify-content-between align-items-center mb-3">
            <div>
                <h5>Transaction Records (<?= number_format($total_records) ?> total)</h5>
            </div>
            <div>
                <button class="btn btn-success" onclick="exportToExcel()">
                    <i class="bi bi-file-earmark-excel"></i> Export to Excel
                </button>
                <button class="btn btn-danger" onclick="exportToPDF()">
                    <i class="bi bi-file-earmark-pdf"></i> Export to PDF
                </button>
                <a href="add_transaction.php<?= $selected_customer ? '?customer=' . urlencode($selected_customer) : '' ?>" class="btn btn-primary">
                    <i class="bi bi-plus-circle"></i> Add Transaction
                </a>
            </div>
        </div>

        <!-- Transactions Table -->
        <div class="table-container">
            <div class="table-responsive">
                <table class="table table-hover" id="transactionsTable">
                    <thead class="table-dark">
                        <tr>
                            <th>ID</th>
                            <th>Date</th>
                            <th>Customer</th>
                            <th>Type</th>
                            <th>Description</th>
                            <th>Foreign Amount</th>
                            <th>AED Amount</th>
                            <th>Balance Before</th>
                            <th>Balance After</th>
                            <th>Notes</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($transactions)): ?>
                            <tr>
                                <td colspan="11" class="text-center text-muted py-4">
                                    <i class="bi bi-inbox" style="font-size: 3rem;"></i>
                                    <p class="mt-2">No transactions found</p>
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($transactions as $trans): ?>
                            <tr class="<?= $trans['transaction_type'] == 'CREDIT' ? 'credit-row' : 'debit-row' ?>">
                                <td><?= htmlspecialchars($trans['id']) ?></td>
                                <td><?= date('Y-m-d H:i', strtotime($trans['transaction_date'])) ?></td>
                                <td>
                                    <strong><?= htmlspecialchars($trans['customer_name'] ?: $trans['customer_phone']) ?></strong><br>
                                    <small class="text-muted"><?= htmlspecialchars($trans['business_type'] ?: '') ?></small>
                                </td>
                                <td>
                                    <span class="badge <?= $trans['transaction_type'] == 'CREDIT' ? 'bg-success' : 'bg-danger' ?>">
                                        <?= htmlspecialchars($trans['transaction_type']) ?>
                                    </span>
                                </td>
                                <td><?= htmlspecialchars($trans['description'] ?: '-') ?></td>
                                <td>
                                    <?php if ($trans['foreign_amount']): ?>
                                        <?= number_format($trans['foreign_amount'], 2) ?> 
                                        <?= htmlspecialchars($trans['foreign_currency']) ?>
                                        <br><small class="text-muted">Rate: <?= $trans['exchange_rate'] ?></small>
                                    <?php else: ?>
                                        -
                                    <?php endif; ?>
                                </td>
                                <td><strong><?= number_format($trans['aed_amount'], 2) ?> AED</strong></td>
                                <td><?= number_format($trans['balance_before'], 2) ?></td>
                                <td><strong><?= number_format($trans['balance_after'], 2) ?></strong></td>
                                <td><?= htmlspecialchars(substr($trans['notes'], 0, 50)) ?><?= strlen($trans['notes']) > 50 ? '...' : '' ?></td>
                                <td>
                                    <button class="btn btn-sm btn-info action-btn" onclick="viewTransaction(<?= $trans['id'] ?>)" title="View">
                                        <i class="bi bi-eye"></i>
                                    </button>
                                    <a href="edit_transaction.php?id=<?= $trans['id'] ?>" class="btn btn-sm btn-warning action-btn" title="Edit">
                                        <i class="bi bi-pencil"></i>
                                    </a>
                                    <button class="btn btn-sm btn-secondary action-btn" onclick="copyTransaction(<?= $trans['id'] ?>)" title="Copy">
                                        <i class="bi bi-clipboard"></i>
                                    </button>
                                    <button class="btn btn-sm btn-danger action-btn" onclick="deleteTransaction(<?= $trans['id'] ?>)" title="Delete">
                                        <i class="bi bi-trash"></i>
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            <?php if ($total_pages > 1): ?>
            <nav aria-label="Transaction pagination">
                <ul class="pagination justify-content-center">
                    <li class="page-item <?= $page <= 1 ? 'disabled' : '' ?>">
                        <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $page - 1])) ?>">Previous</a>
                    </li>
                    
                    <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                        <?php if ($i == 1 || $i == $total_pages || abs($i - $page) <= 2): ?>
                            <li class="page-item <?= $i == $page ? 'active' : '' ?>">
                                <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $i])) ?>"><?= $i ?></a>
                            </li>
                        <?php elseif (abs($i - $page) == 3): ?>
                            <li class="page-item disabled"><span class="page-link">...</span></li>
                        <?php endif; ?>
                    <?php endfor; ?>
                    
                    <li class="page-item <?= $page >= $total_pages ? 'disabled' : '' ?>">
                        <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $page + 1])) ?>">Next</a>
                    </li>
                </ul>
            </nav>
            <?php endif; ?>
        </div>
    </div>

    <!-- View Transaction Modal -->
    <div class="modal fade" id="viewModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Transaction Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" id="viewModalBody">
                    Loading...
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf-autotable/3.5.31/jspdf.plugin.autotable.min.js"></script>
    
    <script>
        function viewTransaction(id) {
            fetch('get_transaction.php?id=' + id)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const t = data.transaction;
                        document.getElementById('viewModalBody').innerHTML = `
                            <div class="row">
                                <div class="col-md-6">
                                    <p><strong>Transaction ID:</strong> ${t.id}</p>
                                    <p><strong>Customer:</strong> ${t.customer_name || t.customer_phone}</p>
                                    <p><strong>Phone:</strong> ${t.customer_phone}</p>
                                    <p><strong>Business Type:</strong> ${t.business_type || 'N/A'}</p>
                                    <p><strong>Transaction Type:</strong> <span class="badge ${t.transaction_type == 'CREDIT' ? 'bg-success' : 'bg-danger'}">${t.transaction_type}</span></p>
                                </div>
                                <div class="col-md-6">
                                    <p><strong>Date:</strong> ${t.transaction_date}</p>
                                    <p><strong>Foreign Amount:</strong> ${t.foreign_amount ? t.foreign_amount + ' ' + t.foreign_currency : 'N/A'}</p>
                                    <p><strong>Exchange Rate:</strong> ${t.exchange_rate || 'N/A'}</p>
                                    <p><strong>AED Amount:</strong> ${t.aed_amount} AED</p>
                                </div>
                                <div class="col-12 mt-3">
                                    <p><strong>Balance Before:</strong> ${t.balance_before} AED</p>
                                    <p><strong>Balance After:</strong> ${t.balance_after} AED</p>
                                    <p><strong>Description:</strong> ${t.description || 'N/A'}</p>
                                    <p><strong>Notes:</strong> ${t.notes || 'N/A'}</p>
                                </div>
                            </div>
                        `;
                        new bootstrap.Modal(document.getElementById('viewModal')).show();
                    } else {
                        alert('Error loading transaction');
                    }
                })
                .catch(error => {
                    alert('Error: ' + error);
                });
        }

        function deleteTransaction(id) {
            if (confirm('Are you sure you want to delete this transaction? This action cannot be undone.')) {
                fetch('delete_transaction.php?id=' + id, {method: 'POST'})
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            alert('Transaction deleted successfully');
                            location.reload();
                        } else {
                            alert('Error: ' + data.message);
                        }
                    })
                    .catch(error => {
                        alert('Error: ' + error);
                    });
            }
        }

        function copyTransaction(id) {
            window.location.href = 'add_transaction.php?copy=' + id;
        }

        function exportToExcel() {
            const table = document.getElementById('transactionsTable');
            const wb = XLSX.utils.table_to_book(table, {sheet: "Transactions"});
            XLSX.writeFile(wb, 'transactions_' + new Date().toISOString().slice(0,10) + '.xlsx');
        }

        function exportToPDF() {
            const { jsPDF } = window.jspdf;
            const doc = new jsPDF('l', 'mm', 'a4');
            
            doc.setFontSize(18);
            doc.text('Customer Transactions Report', 14, 15);
            doc.setFontSize(10);
            doc.text('Generated: ' + new Date().toLocaleString(), 14, 22);
            
            const tableData = [];
            const rows = document.querySelectorAll('#transactionsTable tbody tr');
            
            rows.forEach(row => {
                if (row.cells.length > 1) {
                    const rowData = [
                        row.cells[0].innerText,
                        row.cells[1].innerText,
                        row.cells[2].innerText,
                        row.cells[3].innerText,
                        row.cells[4].innerText,
                        row.cells[6].innerText,
                        row.cells[7].innerText,
                        row.cells[8].innerText
                    ];
                    tableData.push(rowData);
                }
            });
            
            doc.autoTable({
                head: [['ID', 'Date', 'Customer', 'Type', 'Description', 'AED Amount', 'Balance Before', 'Balance After']],
                body: tableData,
                startY: 28,
                styles: { fontSize: 8 },
                headStyles: { fillColor: [102, 126, 234] }
            });
            
            doc.save('transactions_' + new Date().toISOString().slice(0,10) + '.pdf');
        }
    </script>
</body>
</html>
