<?php
// Database connection
$host = 'localhost:3306';
$dbname = 'tawasoll_book';
$username = 'root';
$password = '';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    die("Connection failed: " . $e->getMessage());
}

// Get all customers for dropdown
$customers_query = "SELECT customer_phone, name, business_type, preferred_currency FROM customer_context ORDER BY name";
$customers = $pdo->query($customers_query)->fetchAll(PDO::FETCH_ASSOC);

// Check if copying transaction
$copy_transaction = null;
if (isset($_GET['copy'])) {
    $stmt = $pdo->prepare("SELECT * FROM transactions WHERE id = :id");
    $stmt->execute([':id' => $_GET['copy']]);
    $copy_transaction = $stmt->fetch(PDO::FETCH_ASSOC);
}

// Pre-select customer if provided
$preselected_customer = isset($_GET['customer']) ? $_GET['customer'] : '';
if ($copy_transaction) {
    $preselected_customer = $copy_transaction['customer_phone'];
}

$success_message = '';
$error_message = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    try {
        $pdo->beginTransaction();
        
        $customer_phone = $_POST['customer_phone'];
        $transaction_type = $_POST['transaction_type'];
        $foreign_currency = $_POST['foreign_currency'] ?: null;
        $foreign_amount = $_POST['foreign_amount'] ?: null;
        $exchange_rate = $_POST['exchange_rate'] ?: null;
        $aed_amount = $_POST['aed_amount'];
        $description = $_POST['description'] ?: null;
        $notes = $_POST['notes'] ?: null;
        
        // Get current customer balance
        $customer_stmt = $pdo->prepare("SELECT current_balance FROM customer_context WHERE customer_phone = :phone");
        $customer_stmt->execute([':phone' => $customer_phone]);
        $customer = $customer_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$customer) {
            throw new Exception('Customer not found');
        }
        
        $balance_before = $customer['current_balance'];
        
        // Calculate balance after
        if ($transaction_type == 'CREDIT') {
            $balance_after = $balance_before + $aed_amount;
        } else {
            $balance_after = $balance_before - $aed_amount;
        }
        
        // Insert transaction
        $insert_query = "INSERT INTO transactions 
                        (customer_phone, transaction_date, transaction_type, foreign_currency, 
                         foreign_amount, exchange_rate, aed_amount, balance_before, balance_after, 
                         description, notes)
                        VALUES 
                        (:customer_phone, :transaction_date, :transaction_type, :foreign_currency,
                         :foreign_amount, :exchange_rate, :aed_amount, :balance_before, :balance_after,
                         :description, :notes)";
        
        $insert_stmt = $pdo->prepare($insert_query);
        $insert_stmt->execute([
            ':customer_phone' => $customer_phone,
            ':transaction_date' => date('Y-m-d H:i:s'),
            ':transaction_type' => $transaction_type,
            ':foreign_currency' => $foreign_currency,
            ':foreign_amount' => $foreign_amount,
            ':exchange_rate' => $exchange_rate,
            ':aed_amount' => $aed_amount,
            ':balance_before' => $balance_before,
            ':balance_after' => $balance_after,
            ':description' => $description,
            ':notes' => $notes
        ]);
        
        // Update customer balance and totals
        if ($transaction_type == 'CREDIT') {
            $update_query = "UPDATE customer_context 
                           SET current_balance = current_balance + :amount,
                               total_credits = total_credits + :amount,
                               transaction_count = transaction_count + 1
                           WHERE customer_phone = :phone";
        } else {
            $update_query = "UPDATE customer_context 
                           SET current_balance = current_balance - :amount,
                               total_debits = total_debits + :amount,
                               transaction_count = transaction_count + 1
                           WHERE customer_phone = :phone";
        }
        
        $update_stmt = $pdo->prepare($update_query);
        $update_stmt->execute([
            ':amount' => $aed_amount,
            ':phone' => $customer_phone
        ]);
        
        $pdo->commit();
        
        $success_message = 'Transaction added successfully!';
        
        // Redirect after 2 seconds
        header("refresh:2;url=customer_bookkeeping.php?customer=" . urlencode($customer_phone));
        
    } catch(Exception $e) {
        $pdo->rollBack();
        $error_message = 'Error: ' . $e->getMessage();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add Transaction - Tawasoll Book</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        body {
            background-color: #f8f9fa;
        }
        .header-section {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 2rem 0;
            margin-bottom: 2rem;
        }
        .form-container {
            background: white;
            border-radius: 10px;
            padding: 2rem;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .required::after {
            content: " *";
            color: red;
        }
    </style>
</head>
<body>
    <div class="header-section">
        <div class="container">
            <h1 class="mb-0"><i class="bi bi-plus-circle"></i> Add New Transaction</h1>
            <p class="mb-0">Tawasoll Book - Customer Transaction Entry</p>
        </div>
    </div>

    <div class="container">
        <div class="row justify-content-center">
            <div class="col-lg-8">
                <?php if ($success_message): ?>
                <div class="alert alert-success alert-dismissible fade show">
                    <i class="bi bi-check-circle"></i> <?= $success_message ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <?php if ($error_message): ?>
                <div class="alert alert-danger alert-dismissible fade show">
                    <i class="bi bi-exclamation-triangle"></i> <?= $error_message ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>

                <?php if ($copy_transaction): ?>
                <div class="alert alert-info">
                    <i class="bi bi-info-circle"></i> Copying transaction ID: <?= $copy_transaction['id'] ?>
                </div>
                <?php endif; ?>

                <div class="form-container">
                    <form method="POST" action="" id="transactionForm">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label required">Customer</label>
                                <select name="customer_phone" class="form-select" required id="customerSelect">
                                    <option value="">Select Customer</option>
                                    <?php foreach ($customers as $customer): ?>
                                        <option value="<?= htmlspecialchars($customer['customer_phone']) ?>"
                                                data-currency="<?= htmlspecialchars($customer['preferred_currency']) ?>"
                                                <?= $preselected_customer == $customer['customer_phone'] ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($customer['name'] ?: $customer['customer_phone']) ?> 
                                            (<?= htmlspecialchars($customer['business_type'] ?: 'N/A') ?>)
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label class="form-label required">Transaction Type</label>
                                <select name="transaction_type" class="form-select" required>
                                    <option value="">Select Type</option>
                                    <option value="CREDIT" <?= ($copy_transaction && $copy_transaction['transaction_type'] == 'CREDIT') ? 'selected' : '' ?>>
                                        CREDIT (Money In)
                                    </option>
                                    <option value="DEBIT" <?= ($copy_transaction && $copy_transaction['transaction_type'] == 'DEBIT') ? 'selected' : '' ?>>
                                        DEBIT (Money Out)
                                    </option>
                                </select>
                            </div>

                            <div class="col-md-4 mb-3">
                                <label class="form-label">Foreign Currency</label>
                                <input type="text" name="foreign_currency" class="form-control" 
                                       placeholder="USD, EUR, etc." id="foreignCurrency"
                                       value="<?= $copy_transaction ? htmlspecialchars($copy_transaction['foreign_currency']) : '' ?>">
                            </div>

                            <div class="col-md-4 mb-3">
                                <label class="form-label">Foreign Amount</label>
                                <input type="number" step="0.01" name="foreign_amount" class="form-control" 
                                       placeholder="0.00" id="foreignAmount"
                                       value="<?= $copy_transaction ? htmlspecialchars($copy_transaction['foreign_amount']) : '' ?>">
                            </div>

                            <div class="col-md-4 mb-3">
                                <label class="form-label">Exchange Rate</label>
                                <input type="number" step="0.0001" name="exchange_rate" class="form-control" 
                                       placeholder="0.0000" id="exchangeRate"
                                       value="<?= $copy_transaction ? htmlspecialchars($copy_transaction['exchange_rate']) : '' ?>">
                            </div>

                            <div class="col-md-12 mb-3">
                                <label class="form-label required">AED Amount</label>
                                <div class="input-group">
                                    <input type="number" step="0.01" name="aed_amount" class="form-control" 
                                           placeholder="0.00" required id="aedAmount"
                                           value="<?= $copy_transaction ? htmlspecialchars($copy_transaction['aed_amount']) : '' ?>">
                                    <span class="input-group-text">AED</span>
                                    <button class="btn btn-outline-secondary" type="button" id="calculateBtn">
                                        <i class="bi bi-calculator"></i> Calculate
                                    </button>
                                </div>
                                <small class="text-muted">Enter amount in AED or calculate from foreign currency</small>
                            </div>

                            <div class="col-md-12 mb-3">
                                <label class="form-label">Description</label>
                                <input type="text" name="description" class="form-control" 
                                       placeholder="Brief description of transaction"
                                       value="<?= $copy_transaction ? htmlspecialchars($copy_transaction['description']) : '' ?>">
                            </div>

                            <div class="col-md-12 mb-3">
                                <label class="form-label">Notes</label>
                                <textarea name="notes" class="form-control" rows="3" 
                                          placeholder="Additional notes or comments"><?= $copy_transaction ? htmlspecialchars($copy_transaction['notes']) : '' ?></textarea>
                            </div>
                        </div>

                        <div class="d-flex justify-content-between mt-4">
                            <a href="customer_bookkeeping.php<?= $preselected_customer ? '?customer=' . urlencode($preselected_customer) : '' ?>" 
                               class="btn btn-secondary">
                                <i class="bi bi-arrow-left"></i> Back to List
                            </a>
                            <div>
                                <button type="reset" class="btn btn-outline-secondary">
                                    <i class="bi bi-x-circle"></i> Reset
                                </button>
                                <button type="submit" class="btn btn-primary">
                                    <i class="bi bi-save"></i> Save Transaction
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Auto-populate currency when customer is selected
        document.getElementById('customerSelect').addEventListener('change', function() {
            const selectedOption = this.options[this.selectedIndex];
            const currency = selectedOption.getAttribute('data-currency');
            if (currency) {
                document.getElementById('foreignCurrency').value = currency;
            }
        });

        // Calculate AED amount from foreign currency
        document.getElementById('calculateBtn').addEventListener('click', function() {
            const foreignAmount = parseFloat(document.getElementById('foreignAmount').value) || 0;
            const exchangeRate = parseFloat(document.getElementById('exchangeRate').value) || 0;
            
            if (foreignAmount && exchangeRate) {
                const aedAmount = foreignAmount * exchangeRate;
                document.getElementById('aedAmount').value = aedAmount.toFixed(2);
            } else {
                alert('Please enter both foreign amount and exchange rate');
            }
        });

        // Auto-calculate when foreign amount or exchange rate changes
        document.getElementById('foreignAmount').addEventListener('input', autoCalculate);
        document.getElementById('exchangeRate').addEventListener('input', autoCalculate);

        function autoCalculate() {
            const foreignAmount = parseFloat(document.getElementById('foreignAmount').value) || 0;
            const exchangeRate = parseFloat(document.getElementById('exchangeRate').value) || 0;
            
            if (foreignAmount && exchangeRate) {
                const aedAmount = foreignAmount * exchangeRate;
                document.getElementById('aedAmount').value = aedAmount.toFixed(2);
            }
        }
    </script>
</body>
</html>
